<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <!--  To see the input used for this xslt, check the "Create source.xml file 
        when creating report" menu item under the "Reports" menu in FsComp.
        When checked a .source.xml file is created each time you create a report.
        It shows the actual xml input to the xslt processor.
        This is helpfull if you like to modify this file.
  -->

  <xsl:output method="html" encoding="utf-8" indent="yes" />

  <!--  Set the decimal separator to be used (. or ,) when decimal data is displayed.
        All decimal data in the source is with . and will be displayed with . unless
        formated otherwise using format-number() function.
  -->
  <xsl:variable name="decimal_separator" select="','"/>
  <xsl:decimal-format decimal-separator=',' grouping-separator=' ' />
  <!-- note! make sure both above use same, ie either . or ,!! -->

  <!--  All <xsl:param ... elements will show as a field in a report dialog in 
        FS when creating reports. This means you can define param elements here 
        with a default value and set the value from FS when creating report.
        Some is used simply to display text at the top of the report (ie status), 
        others is used to filter the results.
        If you add "filter" params you must of course also change the "filter"
        definition below so that the filter params is applied.
				20080518 FS 1.2.3: 
				Removed all "filter" params.
				Moved filtering inside FS so the xml input to the xslt is already filtered.
				the filter_info attribute of FsTaskResults element shows what filter(s) is applied.
  -->
  <xsl:param name="title"></xsl:param>
  <xsl:param name="status">Provisional</xsl:param>
  <!-- filter params -->

  <!--  The node-set that this variable returns is what is used to create the result list.
  -->
  <xsl:variable name="filter" select="/Fs/FsCompetition/FsTaskResults[1]/FsParticipant"/>
  <xsl:variable name="task_id" select="/Fs/FsCompetition[1]/FsTaskResults/@id"/>
  <xsl:variable name="filter_info" select="/Fs/FsCompetition[1]/FsTaskResults/@filter_info"/>

  <!-- the FsTask element for the given task (lots of info under there that we need) -->
  <xsl:variable name="task" select="/Fs/FsCompetition[1]/FsTasks/FsTask[@id=$task_id]"/>
  <!-- all participants in the comp -->
  <xsl:variable name="comp_pilots" select="/Fs/FsCompetition[1]/FsParticipants[1]/FsParticipant"/>
  <!-- various stuff we need later ... -->
  <xsl:variable name="tp1_open" select="$task/FsTaskDefinition/FsTurnpoint/@open"/>
  <xsl:variable name="task_name" select="$task/@name"/>
  <xsl:variable name="use_leading_points" select="$task/FsScoreFormula/@use_leading_points"/>
  <xsl:variable name="use_departure_points" select="$task/FsScoreFormula/@use_departure_points"/>
  <xsl:variable name="use_arrival_position_points" select="$task/FsScoreFormula/@use_arrival_position_points"/>
  <xsl:variable name="use_arrival_time_points" select="$task/FsScoreFormula/@use_arrival_time_points"/>
  <xsl:variable name="ss" select="$task/FsTaskDefinition/@ss"/>
  <xsl:variable name="es" select="$task/FsTaskDefinition/@es"/>
  <xsl:variable name="goal_altitude" select="$task/FsTaskState/@goal_altitude"/>
  <xsl:variable name="task_state" select="$task/FsTaskState/@task_state"/>
  <xsl:variable name="cancel_reason" select="$task/FsTaskState/@cancel_reason"/>
  <xsl:variable name="stop_time" select="$task/FsTaskState/@stop_time"/>
  <xsl:variable name="score_back_time" select="$task/FsTaskState/@score_back_time"/>
  <xsl:variable name="task_distance" select="$task/FsTaskScoreParams/@task_distance"/>
  <xsl:variable name="ss_distance" select="$task/FsTaskScoreParams/@ss_distance"/>
  <xsl:variable name="FsTaskScoreParams" select="$task/FsTaskScoreParams"/>
  <xsl:variable name="FsScoreFormula" select="$task/FsScoreFormula"/>
  <xsl:variable name="bonus_gr" select="$task/FsScoreFormula/@bonus_gr"/>
  <xsl:variable name="no_of_startgates" select="count($task/FsTaskDefinition/FsStartGate)"/>
  <xsl:variable name="lcletters">abcdefghijklmnopqrstuvwxyz</xsl:variable>
  <xsl:variable name="ucletters">ABCDEFGHIJKLMNOPQRSTUVWXYZ</xsl:variable>

  <!--  Returns a ranking number based on the @points attribute in the element given by the item param.
        Will give same ranking to elements with equal points. 
				NOTE! Likly to give wrong rank for some pilots when applied to a filtered node-list.
        NOTE! does NOT work when called inside a sorted node-list!!!
  -->
  <xsl:template name="calc_rank_from_points" >
    <xsl:param name="item"/>
    <xsl:param name="points"/>
    <xsl:param name="sub" select="0"/>
    <xsl:variable name="found" select="boolean($item/preceding-sibling::node()[1]/@points = $points)"/>
    <xsl:choose>
      <xsl:when test="$found = true()">
        <xsl:call-template name="calc_rank_from_points">
          <xsl:with-param name="item" select="$item/preceding-sibling::node()[1]"/>
          <xsl:with-param name="points" select="$points"/>
          <xsl:with-param name="sub" select="$sub+1"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="position()-$sub"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- list of startgates -->
  <xsl:template name="FsStartGate_list">
    <xsl:text>StartGate(s): </xsl:text>
    <xsl:value-of select="substring($task/FsTaskDefinition/FsStartGate[1]/@open, 12, 5)"/>
    <xsl:for-each select="$task/FsTaskDefinition/FsStartGate[position() > 1]">
      <xsl:text>, </xsl:text>
      <xsl:value-of select="substring(@open, 12, 5)"/>
    </xsl:for-each>
  </xsl:template>

  <!-- list of scoring formula parameters -->
  <xsl:template name="FsScoreFormula_list">
    <h3 id='scoring'>Scoring formula settings</h3>
    <table class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">param</th>
          <th class="fs_res">value</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="$FsScoreFormula/@*">
          <tr>
            <td class="fs_res">
              <xsl:value-of select="name()"/>
            </td>
            <td class="fs_res" align="right">
              <xsl:value-of select="."/>
            </td>
          </tr>
        </xsl:for-each>
      </tbody>
    </table>
  </xsl:template>

  <!-- task statistics (all sort of intermediate values used to calculate the score of each pilot) -->
  <xsl:template name="FsTaskScoreParams_list">
    <h3 id='taskStatistics'>Task statistics</h3>
    <table class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">param</th>
          <th class="fs_res">value</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="$FsTaskScoreParams/@*">
          <tr>
            <td class="fs_res">
              <xsl:value-of select="name()"/>
            </td>
            <td class="fs_res" align="right">
              <xsl:value-of select="."/>
            </td>
          </tr>
        </xsl:for-each>
      </tbody>
    </table>
  </xsl:template>

  <!-- list pilots with penalties applied to score -->
  <xsl:template name="Penalty_list">
    <h3 id='penalties'>Penalties</h3>
    <i>Note: % penalty is used to calc penalty as a % of total score. Both types can be combined. None affect the scoring of other pilots.</i>
    <table class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">Id</th>
          <th class="fs_res">Name</th>
          <th class="fs_res">% penalty</th>
          <th class="fs_res">points penalty</th>
          <th class="fs_res">Reason</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="$filter">
          <xsl:sort select="@id" data-type="number" />
          <xsl:variable name="pilot_id" select="@id"/>
          <xsl:variable name="penalty" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty"/>
          <xsl:variable name="penalty_points_auto" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty_points_auto"/>
          <xsl:variable name="penalty_points" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty_points"/>
          <xsl:if test="$penalty != 0 or $penalty_points != 0">
            <tr>
              <td class="fs_res" align="right">
                <xsl:value-of select="$pilot_id"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="//FsCompetition[1]/FsParticipants[1]/FsParticipant[@id=$pilot_id]/@name"/>
              </td>
              <td class="fs_res" align="right">
                <xsl:value-of select="$penalty * 100"/>%
              </td>
              <td class="fs_res" align="right">
                <xsl:value-of select="$penalty_points"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty_reason"/>
              </td>
            </tr>
          </xsl:if>
          <xsl:if test="$penalty_points_auto != 0">
            <tr>
              <td class="fs_res" align="right">
                <xsl:value-of select="$pilot_id"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="//FsCompetition[1]/FsParticipants[1]/FsParticipant[@id=$pilot_id]/@name"/>
              </td>
              <td class="fs_res">
              </td>
              <td class="fs_res" align="right">
                <xsl:value-of select="round($penalty_points_auto)"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty_reason_auto"/>
              </td>
            </tr>
          </xsl:if>
        </xsl:for-each>
      </tbody>
    </table>
  </xsl:template>

  <!-- NYP pilots -->
  <xsl:template name="NYP_pilots">
    <h3 id='pilotsNYP'>Pilots not yet processed (NYP)</h3>
    <table class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">Id</th>
          <th class="fs_res">Name</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="//FsCompetition[1]/FsParticipants[1]/FsParticipant">
          <xsl:sort select="@id" data-type="number" />
          <xsl:variable name="pilot_id" select="@id"/>
          <xsl:if test="boolean($task/FsParticipants[1]/FsParticipant[@id=$pilot_id]) = false()">
            <tr>
              <td class="fs_res" align="right">
                <xsl:value-of select="$pilot_id"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="//FsCompetition[1]/FsParticipants[1]/FsParticipant[@id=$pilot_id]/@name"/>
              </td>
            </tr>
          </xsl:if>
        </xsl:for-each>
      </tbody>
    </table>
  </xsl:template>

  <!-- ABS pilots -->
  <xsl:template name="ABS_pilots">
    <h3 id='pilotsABS'>Pilots absent from task (ABS)</h3>
    <table class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">Id</th>
          <th class="fs_res">Name</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="$task/FsParticipants[1]/FsParticipant">
          <xsl:sort select="@id" data-type="number" />
          <xsl:variable name="pilot_id" select="@id"/>
          <xsl:if test="boolean(FsResult) = false()">
            <tr>
              <td class="fs_res" align="right">
                <xsl:value-of select="$pilot_id"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="//FsCompetition[1]/FsParticipants[1]/FsParticipant[@id=$pilot_id]/@name"/>
              </td>
            </tr>
          </xsl:if>
        </xsl:for-each>
      </tbody>
    </table>
  </xsl:template>


  <!-- list pilots with notes -->
  <xsl:template name="Notes_list">
    <h3  id='Notes'>Notes</h3>
    <table class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">Id</th>
          <th class="fs_res">Name</th>
          <th class="fs_res">Note</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="$filter">
          <xsl:sort select="@id"/>
          <xsl:variable name="pilot_id" select="@id"/>
          <xsl:variable name="note" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsFlightDataNote/@note"/>
          <xsl:if test="$note != ''">
            <tr>
              <td class="fs_res" align="right">
                <xsl:value-of select="$pilot_id"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="//FsCompetition[1]/FsParticipants[1]/FsParticipant[@id=$pilot_id]/@name"/>
              </td>
              <td class="fs_res">
                <xsl:value-of select="$note"/>
              </td>
            </tr>
          </xsl:if>
        </xsl:for-each>
      </tbody>
    </table>
  </xsl:template>

  <xsl:template name="turnpointlist">
    <table  id='turnpointlist' class="fs_res">
      <thead>
        <tr>
          <th class="fs_res">No</th>
          <th class="fs_res">Dist.</th>
          <th class="fs_res">Id</th>
          <th class="fs_res">
            Radius<br/>(meters)
          </th>
          <!--th>Type</th-->
          <th class="fs_res">Open</th>
          <th class="fs_res">Close</th>
          <th class="fs_res">Coordinates</th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="$task/FsTaskDefinition/FsTurnpoint">
          <tr>
            <xsl:variable name="position" select="position()"/>
            <xsl:variable name="FsTaskDistToTp"
                      select="$task/FsTaskScoreParams/FsTaskDistToTp[@tp_no=$position]"/>
            <td class="fs_res">
              <xsl:value-of select="$FsTaskDistToTp/@tp_no"/>
              <xsl:if test="$FsTaskDistToTp/@tp_no=$ss">
                <xsl:text> SS</xsl:text>
              </xsl:if>
              <xsl:if test="$FsTaskDistToTp/@tp_no=$es">
                <xsl:text> ES</xsl:text>
              </xsl:if>
            </td>
            <td class="fs_res" align="right">
              <xsl:value-of select="format-number($FsTaskDistToTp/@distance, concat('#0', $decimal_separator, '0'))"/> km
              <!--xsl:value-of select="$FsTaskDistToTp/@distance"/-->
            </td>
            <td class="fs_res">
              <xsl:value-of select="@id"/>
            </td>
            <td class="fs_res" align="right">
              <xsl:value-of select="@radius"/>
            </td>
            <!--td>
              <xsl:value-of select="@type"/>
            </td-->
            <!--
              @open and @close expected on the form: open="2007-05-17T14:00:00+02:00" close="2007-05-17T18:30:00+02:00"
              We only want to show the local time (no date)
            -->
            <td class="fs_res">
              <xsl:value-of select="substring(@open, 12, 5)"/>
            </td>
            <td class="fs_res">
              <xsl:value-of select="substring(@close, 12, 5)"/>
            </td>
            <td class="fs_res">
              <xsl:choose>
                <xsl:when test="@utm_zone">
                  <xsl:text> </xsl:text>
                  <xsl:value-of select="@utm_zone"/>
                  &#160;<xsl:value-of select="@lon"/>
                  &#160;<xsl:value-of select="@lat"/>
                </xsl:when>
                <xsl:otherwise>
                  Lat: <xsl:value-of select="@lat"/> Lon: <xsl:value-of select="@lon"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:for-each>
      </tbody>
    </table>
    <!--svg xmlns="http://www.w3.org/2000/svg" version="1.1">
  <path d="M45.85352 6.2235 L 45.84882 6.214220 Z"/>
</svg>
<svg xmlns="http://www.w3.org/2000/svg" version="1.1">
  <path d="M150 0 L75 200 L225 200 Z" />
</svg--> 
  </xsl:template>

  <!-- Result list heading row -->
  <xsl:template name="result_heading_row">
    <tr  id="headingrow" >
      <th class="fs_res">#</th>
      <th class="fs_res nomobile">Id</th>
      <th class="fs_res"></th>
      <th class="fs_res">Name</th>
      <th class="fs_res"></th>
      <!-- <th class="fs_res">Nat</th-->
      <th class="fs_res">Glider</th>
      <th class="fs_res nomobile">Sponsor</th>
      <!-- If Race or Elapsed time? -->
      <xsl:if test="$es != ''">
        <th class="fs_res nomobile">SS</th>
        <th class="fs_res nomobile">ES</th>
        <th class="fs_res">Time</th>
        <th class="fs_res nomobile">km/h</th>
      </xsl:if>
      <!-- Stopped task? -->
      <xsl:choose>
        <xsl:when test="$task_state = 'STOPPED' and $bonus_gr > 0">
          <th class="fs_res">
            Last<br/>Dist.<sup>1</sup>
          </th>
          <th class="fs_res">
            Alt.<sup>2</sup>
          </th>
          <th class="fs_res">
            Dist.<sup>3</sup>
          </th>
        </xsl:when>
        <xsl:otherwise>
          <th class="fs_res">Dist.</th>
        </xsl:otherwise>
      </xsl:choose>
      <th class="fs_res">
        Dist.<br/>Points
      </th>
      <xsl:if test="$use_leading_points=1">
        <th class="fs_res">
          Lead.<br/>Points
        </th>
      </xsl:if>
      <xsl:if test="$use_departure_points=1">
        <th class="fs_res">
          Dept.<br/>Points
        </th>
      </xsl:if>
      <th class="fs_res">
        Time<br/>Points
      </th>
      <xsl:if test="$use_arrival_time_points=1">
        <th class="fs_res">
          Arr.<br/>Time<br/>Points
        </th>
      </xsl:if>
      <xsl:if test="$use_arrival_position_points=1">
        <th class="fs_res">
          Arr.<br/>Pos<br/>Points
        </th>
      </xsl:if>
      <th class="fs_res">Total</th>
    </tr>
  </xsl:template>

  <!--  Result list row.
        node-set elements must have @id and @points attributes and be sorted descending on @points.
        Gets other data from the $comp_pilots and $task variables.
  -->
  <xsl:template name="result_row">
    <tr >

      <xsl:variable name="pilot_id" select="@id"/>
      <!-- General pilot info (name, nation, etc ...) -->
      <xsl:variable name="comp_pilot" select="$comp_pilots[@id=$pilot_id]"/>
      <!-- Info about the pilot's task performance (distance, time, etc ...) given by the scoring program. -->
      <xsl:variable name="task_pilot_result" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult"/>
      <xsl:variable name="gender">
        <xsl:choose>
            <xsl:when test="$comp_pilot/@female=1">woman</xsl:when>
            <xsl:otherwise>man</xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      
      <xsl:variable name="category">
        <xsl:value-of select="$comp_pilot/FsCustomAttributes/FsCustomAttribute[@name='Certification']/@value"/>       
      </xsl:variable>  
	  	  
	  <!-- END OF VARIABLES -->
		
	  <!-- show all custom attributes -->
	   
	  <xsl:for-each select="$comp_pilot/FsCustomAttributes/FsCustomAttribute">
		 <xsl:variable name="customattr"><xsl:value-of select="@name"/></xsl:variable>
		 <xsl:attribute name="data-{$customattr}"><xsl:value-of select="@value"/></xsl:attribute>
	  </xsl:for-each>
	  
				
	  <xsl:attribute name="class">fs_res_res_row rowResult</xsl:attribute>
	  
	  <xsl:attribute name="data-nat"><xsl:value-of select="$comp_pilot/FsCustomAttributes/FsCustomAttribute[@name='iso2']/@value"/></xsl:attribute>
	  <xsl:attribute name="data-gender"><xsl:value-of select="$gender"/></xsl:attribute>
	  <xsl:attribute name="data-cat"><xsl:value-of select="$category"/></xsl:attribute>
	  
	  
      <td class="fs_res cellRank" align="right">
        <xsl:call-template name="calc_rank_from_points">
          <xsl:with-param name="item" select="."/>
          <xsl:with-param name="points" select="@points"/>
        </xsl:call-template>
      </td>
      <td class="fs_res cellPilotID nomobile">
        <xsl:value-of select="@id"/>
      </td>
      <td class="fs_res cellFlag">
        <img><xsl:attribute name="src">http://www.livetrack24.com/img/flags/32/<xsl:value-of select="$comp_pilot/FsCustomAttributes/FsCustomAttribute[@name='iso2']/@value"/>.png</xsl:attribute>
        </img>
      </td>
      <td class="fs_res cellName">
        <xsl:value-of select="$comp_pilot/@name"/>
      </td>
      <td class="fs_res cellGender">
        <xsl:choose>
          <xsl:when test="$comp_pilot/@female=1">F</xsl:when>
          <xsl:otherwise>M</xsl:otherwise>
        </xsl:choose>
      </td>
      <!--td class="fs_res">
        <xsl:choose>
          <xsl:when test="$comp_pilot/@female=1">F</xsl:when>
          <xsl:otherwise>M</xsl:otherwise>
        </xsl:choose>
      </td>
      <td class="fs_res">
        <xsl:value-of select="$comp_pilot/@nat_code_3166_a3"/>
      </td-->
      <td class="fs_res cellGlider">
        <xsl:value-of select="$comp_pilot/@glider"/> (<xsl:value-of select="$category"/>)
        
      </td>
      <td class="fs_resc ellSponsor nomobile">
        <xsl:value-of select="$comp_pilot/@sponsor"/>
      </td>
      <!-- If Race or Elapsed time? -->
      <xsl:if test="$es != ''">
        <td class="fs_res nomobile">
          <xsl:value-of select="substring($task_pilot_result/@started_ss, 12, 8)"/>
        </td>
        <td class="fs_res nomobile">
          <xsl:value-of select="substring($task_pilot_result/@finished_ss, 12, 8)"/>
        </td>
        <td class="fs_res">
          <xsl:if test="$task_pilot_result/@finished_ss != ''">
            <xsl:value-of select="$task_pilot_result/@ss_time"/>
          </xsl:if>
        </td>
        <td class="fs_res nomobile">
          <xsl:if test="$task_pilot_result/@finished_ss != ''">
            <xsl:value-of select="format-number($ss_distance div $task_pilot_result/@ss_time_dec_hours, concat('#0', $decimal_separator, '0'))"/>
          </xsl:if>
        </td>
      </xsl:if>
      <xsl:if test="$task_state = 'STOPPED' and $bonus_gr > 0">
        <td class="fs_res" align="right">
          <xsl:value-of select="format-number($task_pilot_result/@last_distance, concat('#0', $decimal_separator, '00'))"/>
        </td>
        <td class="fs_res" align="right">

          <xsl:value-of select="$task_pilot_result/@last_altitude_above_goal"/>
        </td>
      </xsl:if>
      <td class="fs_res" align="right">
        <xsl:choose>
          <xsl:when test="@no_distance != ''">
            <xsl:value-of select="@no_distance"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="format-number($task_pilot_result/@distance, concat('#0', $decimal_separator, '00'))"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td class="fs_res" align="right">
        <xsl:value-of select="format-number($task_pilot_result/@distance_points, concat('#0', $decimal_separator, '0'))"/>
      </td>
      <xsl:if test="$use_leading_points=1">
        <td class="fs_res" align="right">
          <xsl:if test="$task_pilot_result/@leading_points != 0">
            <xsl:value-of select="format-number($task_pilot_result/@leading_points, concat('#0', $decimal_separator, '0'))"/>
          </xsl:if>
        </td>
      </xsl:if>
      <xsl:if test="$use_departure_points=1">
        <td class="fs_res" align="right">
          <xsl:if test="$task_pilot_result/@departure_points != 0">
            <xsl:value-of select="format-number($task_pilot_result/@departure_points, concat('#0', $decimal_separator, '0'))"/>
          </xsl:if>
        </td>
      </xsl:if>
      <td class="fs_res" align="right">
        <xsl:if test="$task_pilot_result/@time_points != 0">
          <xsl:value-of select="format-number($task_pilot_result/@time_points, concat('#0', $decimal_separator, '0'))"/>
        </xsl:if>
      </td>
      <xsl:if test="$use_arrival_time_points=1">
        <td class="fs_res" align="right">
          <xsl:if test="$task_pilot_result/@arrival_points != 0">
            <xsl:value-of select="format-number($task_pilot_result/@arrival_points, concat('#0', $decimal_separator, '0'))"/>
          </xsl:if>
        </td>
      </xsl:if>
      <xsl:if test="$use_arrival_position_points=1">
        <td class="fs_res" align="right">
          <xsl:if test="$task_pilot_result/@arrival_points != 0">
            <xsl:value-of select="format-number($task_pilot_result/@arrival_points, concat('#0', $decimal_separator, '0'))"/>
          </xsl:if>
        </td>
      </xsl:if>
      <td class="fs_res cellPoints" align="right">
		<xsl:attribute name="data-points"><xsl:value-of select="@points"/></xsl:attribute>
	  
        <xsl:variable name="penalty" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty"/>
        <xsl:variable name="penalty_points_auto" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty_points_auto"/>
        <xsl:variable name="penalty_points" select="$task/FsParticipants[1]/FsParticipant[@id=$pilot_id]/FsResult/@penalty_points"/>
        <xsl:choose>
          <xsl:when test="$penalty > 0 or $penalty_points > 0 or $penalty_points_auto > 0">
            <span style="color:red">
              <xsl:value-of select="format-number(@points, '#0')"/>
            </span>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="format-number(@points, '#0')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
    </tr>
  </xsl:template>

  <!-- Main template. This is where it all starts. -->
  <xsl:template match="/">
  <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE html&gt;</xsl:text>
    <html>
      <head>
		<script src="https://code.jquery.com/jquery-1.12.0.min.js"></script>
				
		<link href="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.6/css/bootstrap.min.css" rel="stylesheet" integrity="sha256-7s5uDGW3AHqw6xtJmNNtr+OBRJUlgkNJEo78P4b0yRw= sha512-nNo+yCHEyn0smMxSswnf/OnX6/KwJuZTlNZBjauKhTK0c+zT+q5JOCx0UFhXQ6rJR9jg6Es8gPuD2uZcYDLqSw==" crossorigin="anonymous"></link>

		<script src="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.6/js/bootstrap.min.js" integrity="sha256-KXn5puMvxCw+dAYznun+drMdG1IFl3agK0p/pqT9KAo= sha512-2e8qq0ETcfWRI4HJBzQiA3UoyFk6tbNyG+qSaIBZLyW9Xf3sWZHN/lxe9fTh1U45DpPf07yj94KsUHHWe4Yk1A==" crossorigin="anonymous"></script>

		

<script language="javascript">
var filters={
	'nat':{ 'allText':'All Nations' , 'text':'Nation'} ,
	'gender':{ 'allText':'Male and Female', 'text':'' } ,
	'cat':{ 'allText':'All Categories', 'text':'Category' }
  <xsl:for-each select="/Fs/FsCompetition[1]/FsFilterAttributes/FsFilterAttribute">
    ,'<xsl:value-of select="@name"/>':{ 'allText':'All <xsl:value-of select="@name"/>', 'text':'<xsl:value-of select="@name"/>' } 
  </xsl:for-each>
} ;


</script>

<script src="results_fs_init.js" ></script>
				
<style type="text/css">
				
@media print
{    
    .no-print, .no-print *
    {
        display: none !important;
    }
}

.hover { /* for IE using onmouseover and onmouseout */
  background: red;
  color:white;
}
tr.fs_res_res_row:hover {
  background: red;
  color:white;
}

.gen_woman {
  background: pink;
}
.gen_man {
  background: lightblue;
}
  
tr.odd_man {
  background: #CEEBF5;
}
tr.odd_woman {
  background: #F9D8DE;
}

div.fs_res   {
  font-family: Verdana, Arial, Helvetica, sans-serif;
  font-size: x-small;
}
  
table.fs_res   {
  border:solid 1px gray;
  border-collapse:collapse;
  font-size: x-small;
}
  
td.fs_res   {
  border:solid 1px gray;
  vertical-align:top;
  padding:5px;
}
  
th.fs_res   {
  border:solid 1px gray;
  vertical-align:center;
}
  
.GoLarge{
	height:50px;
	width:213px;
	font-size:large;
}

#countryButtonsDiv {
	width:100%;
}

.cellName img {
	margin-right:4px;
	width: 18px;
}

.cellFlag img {
  margin-right:1px;
  width: 18px;
}

img.btn{
	width:46px;
}

.filterButtons { 
	padding: 13px;
    width: 800px;
    background-color: aliceblue;
}

.filterButtons  div { 
	margin-bottom:10px;
}
</style>
				

      </head>
      <body>
        <div class="fs_res">
			<img><xsl:attribute name="src">http://www.livetrack24.com/files/events/<xsl:value-of select="/Fs/FsCompetition/@eventcode"/>/banner.jpg</xsl:attribute>
			</img> 
			<div id="filterButtonsDiv" class='no-print filterButtons'>
				<div id="countryButtonsDiv"></div>
				<div id="genderButtonsDiv"></div>				
			</div>         
        </div>
        <div>
          <div class="fs_res">
            <div class="fs_res" style="width:100%;font-size: xx-small;text-align:right;" >
              <p style="font-style:italic;">
                Report created: <xsl:value-of select="/Fs/FsCompetition[1]/FsTaskResults/@ts"/>
              </p>
            </div>
            <h2>
              <xsl:value-of select="/Fs/FsCompetition[1]/@name"/>
            </h2>
            <p style="font-size:xx-small">
              <xsl:value-of select="/Fs/FsCompetition[1]/@from"/> to <xsl:value-of select="/Fs/FsCompetition[1]/@to"/>
            </p>
            <xsl:if test="string-length($title) > 0">
              <h2>
                <xsl:value-of select="$title"/>
              </h2>
            </xsl:if>
            <h3>
              <xsl:value-of select="substring($tp1_open, 1, 10)"/>&#160;<xsl:value-of select="$task_name"/> <span id="subtitle">(All pilots)</span>
              <xsl:choose>
                <xsl:when test="$task_state = 'STOPPED'">
                  &#160;-&#160;Stopped&#160;at&#160;<xsl:value-of select="substring($stop_time, 12, 5)"/>
                  <xsl:if test="$score_back_time > 0">
                    &#160;(scored back by <xsl:value-of select="$score_back_time"/> min.)
                  </xsl:if>
                </xsl:when>
                <xsl:when test="$task_state = 'CANCELLED'">
                  &#160;-&#160;Cancelled:&#160;<xsl:value-of select="$cancel_reason"/>
                </xsl:when>
                <xsl:otherwise/>
              </xsl:choose>
            </h3>
            <h4>
              <xsl:choose>
                <xsl:when test="$es and $no_of_startgates > 0">
                  <xsl:text>Race to Goal</xsl:text>
                </xsl:when>
                <xsl:when test="$es and $no_of_startgates = 0">
                  <xsl:text>Elapsed time</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:text>Open Distance</xsl:text>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:if test="$es != ''">
                <xsl:text>&#160;</xsl:text>
                <xsl:value-of select="format-number($task_distance, concat('#0', $decimal_separator, '0'))"/>
                <xsl:text> km</xsl:text>
              </xsl:if>
            </h4>
            <p>
              <xsl:value-of select="$status"/>
            </p>
            <xsl:if test="string-length($filter_info) > 0">
              <p>
                <b>
                  Results include only those pilots where <xsl:value-of select="$filter_info"/>
                </b>
              </p>
            </xsl:if>
            <xsl:call-template name="turnpointlist"/>
            <xsl:if test="$no_of_startgates > 0">
              <xsl:call-template name="FsStartGate_list"/>
            </xsl:if>
            <br/>
            <br/>
            <!-- result list -->
            <table class="fs_res">
              <!-- headings -->
              <thead>
                <xsl:call-template name="result_heading_row"/>
              </thead>
              <!-- loop through the filtered list of pilots -->
              <xsl:for-each select="$filter">
                <xsl:call-template name="result_row"/>
              </xsl:for-each>
            </table>
            <!-- Remark if stopped task -->
            <xsl:if test="$task_state = 'STOPPED' and $bonus_gr > 0">
              <p>
                <div style="font-size:8">
                  <sup>1</sup>: Distance at Task Stop Time (<xsl:value-of select="substring($stop_time, 12, 5)"/>) - Score-back Time (<xsl:value-of select="$score_back_time"/> min.)<br/>
                  <sup>2</sup>: Altitude above goal (<xsl:value-of select="$goal_altitude"/>m) at Task Stop Time - Score-back Time. '0' means pilot landed before the stop time or was below goal altitude.<br/>
                  <sup>3</sup>: Last distance + Altitude * Bonus Glide Ratio. If best distance previously reached is bigger than this, use best distance instead.
                </div>
              </p>
            </xsl:if>
            <!-- List of pilots with notes -->
            <xsl:if test="count($filter[@has_note=1]) > 0">
              <xsl:call-template name="Notes_list"/>
            </xsl:if>
            <!-- List of pilots with penalties -->
            <xsl:if test="count($filter[@has_penalty=1]) > 0">
              <xsl:call-template name="Penalty_list"/>
            </xsl:if>

            <div class="fs_res" style="page-break-before: always;">
              <!-- List ABS pilots -->
              <xsl:call-template name="ABS_pilots"/>

              <!-- List NYP pilots -->
              <xsl:call-template name="NYP_pilots"/>
            </div>

            <!-- task statistics and scoring formula on a new page -->
            <div class="fs_res nomobile" style="page-break-before: always;">
              <xsl:call-template name="FsTaskScoreParams_list"/>
              <xsl:call-template name="FsScoreFormula_list"/>
            </div>
          </div>
        </div>
        <div>
          <table width="640">
            <tr><td>
 
 

</td></tr>
</table>
        </div>
      </body>
    </html>
  </xsl:template>

</xsl:stylesheet>